package com.aston;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.ComboDialogField;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.DialogField;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.IDialogFieldListener;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.LayoutUtil;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.SelectionButtonDialogFieldGroup;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import com.aston.utils.Utils;

/**
 * This is a New Type Wizard.
 * 
 * It simply carries the combo that shows available templates and the group of checks box. <br>
 * 
 * This wizard will create a Java Class. <br>
 * 
 * Part of the <a href="http://renaud91.free.fr/Plugins>Aston Wizard</a><br>
 *
 * @author 
 * <ul>
 * 	<li><a href="mailto:renaud91@free.fr">Ferret Renaud</a></li>
 * </ul>
 * 
 * @version 1.0
 * @since Eclipse 2.0
 */ 
public abstract class NewTypeWizardPage
	extends org.eclipse.jdt.ui.wizards.NewTypeWizardPage
	implements IDialogFieldListener, Marks {

	protected final static String SETTINGS_CREATE_CONSTR = "create_constructor";
	protected final static String SETTINGS_CREATE_INHERITED =
		"create_unimplemented";
	protected final static String SETTINGS_CREATE_MAIN = "create_main";
	protected final static String SETTINGS_TEMPLATE = "template";	

	protected Map templatesMap;
	protected SelectionButtonDialogFieldGroup fMethodStubsButtons;
	protected ComboDialogField templateList;

	/**
	 * Constructor of the object.
	 * 
	 * @param isClass <code>true</code> if a new class is to be created; otherwise
	 * an interface is to be created
	 * 
	 * @param pageName the wizard page's name
	 * @param templateKey the template key
	 */
	public NewTypeWizardPage(boolean isClass, String pageName, String templateKey) {
		super(isClass, pageName);
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeWizardPage.NewTypeWizardPage(isClass, pageName, templateKey)");
		this.templatesMap =
			AstonWizardsPlugin.getDefault().loadTemplateFor(templateKey);				
		initThePage();
	}

	/**
	 * The wizard owning this page is responsible for calling this method 
	 * with the current selection. The selection is used to 
	 * initialize the fields of the wizard page.
	 * 
	 * @param selection used to initialize the fields
	 */
	public void init(IStructuredSelection selection) {
		LogMessage.getInstance().log("-> NewTypeWizardPage.init(selection)");
		
		IJavaElement jelem = getInitialJavaElement(selection);
		initContainerPage(jelem);
		initTypePage(jelem);
		doStatusUpdate();

		boolean[] values = new boolean[3];
		for (int i = 0; i < values.length; i++) {
			values[i] = true;
		}

		IDialogSettings section =
			getDialogSettings().getSection(this.getName());

		if (section != null) {
			values[0] = section.getBoolean(SETTINGS_CREATE_INHERITED);
			values[1] = section.getBoolean(SETTINGS_CREATE_MAIN);
			values[2] = section.getBoolean(SETTINGS_CREATE_CONSTR);
		}
		setMethodStubSelection(values, true);
	}

	/**
	 * Returns the current selection state of the 'Create inherited' abstract 
	 * methods checkbox.
	 * 
	 * @return the selection state of the 'Create inherited' abstract methods
	 * checkbox
	 */
	public boolean isCreateInherited() {
		LogMessage.getInstance().log("-> NewTypeWizardPage.isCreateMain()");
		return this.fMethodStubsButtons.isSelected(0);
	}

	/**
	 * Returns the current selection state of the 'Create Main' checkbox.
	 * 
	 * @return the selection state of the 'Create Main' checkbox
	 */
	public boolean isCreateMain() {
		LogMessage.getInstance().log("-> NewTypeWizardPage.isCreateMain()");
		return this.fMethodStubsButtons.isSelected(1);
	}

	/**
	 * Returns the current selection state of the 'Create Constructors' checkbox.
	 * 
	 * @return the selection state of the 'Create Constructors' checkbox
	 */
	public boolean isCreateConstructors() {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.isCreateConstructors()");
		return this.fMethodStubsButtons.isSelected(2);
	}

	/**
	 * Updates status. 
	 */
	protected void doStatusUpdate() {
		LogMessage.getInstance().log("-> NewTypeWizardPage.doStatusUpdate()");
		setPageComplete(true);
		setErrorMessage(null);
		
		// status of all used components
		IStatus[] status =
			new IStatus[] {
				this.fContainerStatus,
				isEnclosingTypeSelected()
					? this.fEnclosingTypeStatus
					: this.fPackageStatus,
					this.fTypeNameStatus,
					this.fModifierStatus,
					this.fSuperClassStatus,
					this.fSuperInterfacesStatus };

		// the mode severe status will be displayed and the ok 
		// button enabled/disabled.
		updateStatus(status);
		
		if (isPageComplete()) {
			if (this.templateList != null) {
				String template = this.templateList.getText();
				if (template.trim().length() == 0) {
					setErrorMessage(
						AstonWizardsPlugin.getResourceString(
							"lg.warning.templatenamemustbespecified"));
					setPageComplete(false);
					return;
				}
				if (this.templatesMap.get(template.trim()) == null) {
					setErrorMessage(
						AstonWizardsPlugin.getResourceString(
							"lg.warning.templateisnotavailable",
							template));
					setPageComplete(false);								
					return;
				}
			}						
		}	
	}

	/**
	 * Updates status when a field changes.
	 * 
	 * @param fieldName the name of the field that had change
	 */
	protected void handleFieldChanged(String fieldName) {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.handleFieldChanged(fieldName)");
		super.handleFieldChanged(fieldName);

		doStatusUpdate();
	}

	/**
	 * Creates the window that will represent the wizard page. 
	 * 
	 * @param parent the parent of this page
	 */
	public void createControl(Composite parent) {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.createControl(parent)");
		initializeDialogUnits(parent);

		Composite composite = new Composite(parent, SWT.NONE);

		int nColumns = 4;

		GridLayout layout = new GridLayout();
		layout.numColumns = nColumns;
		composite.setLayout(layout);

		// pick & choose the wanted UI components

		createContainerControls(composite, nColumns);
		createPackageControls(composite, nColumns);
		createEnclosingTypeControls(composite, nColumns);

		createSeparator(composite, nColumns);

		createTypeNameControls(composite, nColumns);
		createModifierControls(composite, nColumns);

		createSuperClassControls(composite, nColumns);

		createSuperInterfacesControls(composite, nColumns);

		createControlForTemplate(composite, nColumns);

		createMethodStubSelectionControls(composite, nColumns);

		setControl(composite);
	}

	/**
	 * Creates the label for the template list.
	 * 
	 * @param composite the parent
	 * @param nColumns the number of columns
	 */
	protected void createControlForTemplate(
		Composite composite,
		int nColumns) {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeWizardPage.createControlForTemplate(composite, nColumns)");
		this.templateList =
			new ComboDialogField(SWT.READ_ONLY | SWT.BORDER | SWT.SINGLE);
		this.templateList.setItems(Utils.asStringArray(this.templatesMap, true));
		this.templateList.setDialogFieldListener(this); // Keep this here
		this.templateList.selectItem(0);
		this.templateList.setLabelText(
			AstonWizardsPlugin.getResourceString("lg.label.temptouse"));
		this.templateList.doFillIntoGrid(composite, nColumns);
	}

	/**
	 * Initialize the page.
	 * 
	 * Sets the title, builds the labels ...
	 * @param key the template identification key
	 */
	protected abstract void initThePage();

	/**
	 * Generates the java file. 
	 * 
	 * @param type type of the object
	 * @param imports the imports of the file
	 * @param monitor the progress bar
	 * 
	 * @throws CoreException if an error occurred
	 */
	protected void createTypeMembers(
		IType type,
		ImportsManager imports,
		IProgressMonitor monitor)
		throws CoreException {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeWizardPage.createTypeMembers(type, imports, monitor)");
		Map values = new HashMap(5);

		values.put(MARK_TEMPLATE_NAME, getTemplateFileName());
		values.put(MARK_CLASS_NAME, getTypeName());
		values.put(MARK_PACKAGE_NAME, getPackageText());		

		this.setSpecificValues(type, imports, monitor, values);

		this.createInternal(type, imports, monitor, values);
	}

	/**
	 * Adds specfic values to the Map of keys.
	 * 
	 * @param type type of the object
	 * @param imports the imports of the file
	 * @param monitor the progress bar
	 * @param currentValues the current Map that contains values for the template
	 */
	protected abstract void setSpecificValues(
		IType type,
		ImportsManager imports,
		IProgressMonitor monitor,
		Map currentValues);

	/**
	 * Sets the selection state of the method stub checkboxes.
	 * 
	 * @param values an array of boolean used for options
	 * @param canBeModified indicates if the checked panels can be modified or not
	 */
	public void setMethodStubSelection(
		boolean[] values,
		boolean canBeModified) {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeWizardPage.setMethodStubSelection(values, canBeModified)");
		for (int i = 0; i < values.length; i++) {
			this.fMethodStubsButtons.setSelection(i, values[i]);
		}
		this.fMethodStubsButtons.setEnabled(canBeModified);
	}

	/**
	 * Sets the "old" value in the window.
	 */
	protected void loadSelectionValues() {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.loadSelectionValues()");
		IDialogSettings section =
			getDialogSettings().getSection(this.getName());
		if (section!=null) {			
			String temp = section.get(SETTINGS_TEMPLATE);
		
			if (this.templatesMap!=null &&
				this.templateList!=null && 
				temp!=null && 
				!"".equals(temp)) {	
				this.templateList.setText(temp);
			}
		}		
	}

	/**
	 * Makes the page visible. 
	 * 
	 * @param visible true will show the page, false will hide it
	 */
	public void setVisible(boolean isVisible) {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.setVisible(isVisible)");
		if (isVisible) {
			setFocus();
			this.loadSelectionValues();			
		}
		super.setVisible(isVisible);
	}

	/**
	 * Generates the UI checkboxes for array options. 
	 * 
	 * @param composite where to add the chekboxes
	 * @param nColumns number of checkboxes
	 */
	protected void createMethodStubSelectionControls(
		Composite composite,
		int nColumns) {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeWizardPage.createMethodStubSelectionControls(composite, nColumns)");
		Control labelControl = this.fMethodStubsButtons.getLabelControl(composite);
		LayoutUtil.setHorizontalSpan(labelControl, nColumns);

		DialogField.createEmptySpace(composite);

		Control buttonGroup =
			this.fMethodStubsButtons.getSelectionButtonsGroup(composite);
		LayoutUtil.setHorizontalSpan(buttonGroup, nColumns - 1);
	}

	/**
	 * Returns the template file name
	 * 
	 * @return the template file name
	 */
	public String getTemplateFileName() {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.getTemplateFileName()");
		if (this.templatesMap != null && this.templateList != null) {
			return (String) this.templatesMap.get(this.templateList.getText());
		}
		return null;
	}

	/**
	 * Finds an information in the template
	 * 
	 * @param file the file body. The file should be Java compilant, 
	 * 		   means it should compile.
	 * @param markValue should be a mark
	 * 
	 * @return The value contained between typeStart and typeEnd
	 */
	public final String findXmlPart(String file, final String markValue) {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.findXmlPart(file, markValue)");
		String typeStart = "<".concat(markValue);
		String typeEnd = "</".concat(markValue);
		int start = file.indexOf(typeStart);
		if (start != -1) {
			start = file.indexOf(">", start + typeStart.length());
			start += 1;
			int end = file.indexOf(typeEnd, start);

			if (end < 0) {
				Utils.showError(
					AstonWizardsPlugin.getResourceString(
						"lg.error.nonendtag",
						typeStart),
					getShell());
				return null;
			}
			return file.substring(start, end);
		}
		return null;
	}

	/**
	 * Cuts the template file and find the searched type parts.
	 * 
	 * @param file the file body. The file should be Java compilant, 
	 * 				means it should compile.
	 * @param markValue should be MARK_CONSTRUTOR, MARK_FIELD, MARK_METHOD, ...
	 * @param token should be "name" or "value"
	 * 
	 * @return a SortedMap with key=name of the bloc, and value=the bloc
	 */
	public final Map loadXmlParts(
		String file,
		final String markValue,
		String token) {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.loadXmlParts(file, markValue, token)");
		Map resultat = new TreeMap();
		String typeStart = "<".concat(markValue);
		String typeEnd = "</".concat(markValue);
		int start = file.indexOf(typeStart, 0);
		while (start >= 0) {
			int endT = start + typeStart.length();
			String tokenValue = null;
			if (token != null) {
				// search for token start
				int startT = file.indexOf(token, start + typeStart.length());
				if (startT >= 0) {
					startT = file.indexOf("=\"", start);
					if (startT < 0) {
						Utils.showError(
							AstonWizardsPlugin.getResourceString(
								"lg.error.notoken",
								token,
								typeStart),
							getShell());
						return resultat;
					}
					startT += 2; // for '="'

					endT = file.indexOf("\"", startT);
					tokenValue = file.substring(startT, endT);
				}
			}
			start = file.indexOf(">", endT) + 1; // +1 for '>'

			int end = file.indexOf(typeEnd, start);
			if (end < 0) {
				Utils.showError(
					AstonWizardsPlugin.getResourceString(
						"lg.error.nonendtag",
						typeStart),
					getShell());
				return resultat;
			}

			String body = file.substring(start, end);
			if (tokenValue == null) {
				tokenValue = String.valueOf(body.hashCode());
			}
			resultat.put(tokenValue, body);
			start = file.indexOf(typeStart, end);
		}
		return resultat;
	}

	/**
	 * Refresh the super class field
	 * 
	 * @param force if true will change the actual value of the super class field
	 */
	public void refreshSuperClass(boolean force) {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.refreshSuperClass(force)");
		String sc = super.getSuperClass();
		// Always do it
		if (force) {
			sc = getSuperClassFromTemplate();
			super.setSuperClass(sc, true);
		}
		// Do it only if no super class is specified
		if (!force && sc == null || "".equals(sc.trim())) {
			sc = getSuperClassFromTemplate();
			super.setSuperClass(sc, true);
		}
	}

	/**
	 * Refresh the interface list field
	 * 
	 * @param force if true will change the actual value of the list field
	 */
	protected void refreshInterfaces(boolean force) {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.refreshInterfaces(force)");
		List sc = super.getSuperInterfaces();
		// Always do it
		if (force) {
			sc = getInterfacesFromTemplate();
			super.setSuperInterfaces(sc, true);
		}
		// Do it only if no super class is specified
		if (!force && sc == null || sc.isEmpty()) {
			sc = getInterfacesFromTemplate();
			super.setSuperInterfaces(sc, true);
		}
	}

	/**
	 * Loads the list of interface from the selected template.
	 * 
	 * @return a List that contains all interfaces implemented by the future object
	 */
	protected List getInterfacesFromTemplate() {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.getInterfacesFromTemplate()");
		String tName = getTemplateFileName();
		List resu = new ArrayList(5);
		if (tName != null) {
			StringBuffer res;
			try {
				res = AstonWizardsPlugin.getDefault().buildFromTemplate(tName);
			} catch (IOException e) {
				return null;
			}
			Map m =
				loadXmlParts(res.toString(), MARK_IMPLEMENTS_INTERFACE, null);
			resu.addAll(m.values());
			m.clear();
			m = null;
			res = null;
		}
		return resu;
	}

	/**
	 * Loads the super class name from the selected template.
	 * 
	 * @return the super class name defined in the template, or java.lang.Object if none found
	 */

	protected String getSuperClassFromTemplate() {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.getSuperClassFromTemplate()");
		String tName = getTemplateFileName();
		String resu = "java.lang.Object";
		if (tName != null) {
			StringBuffer res;
			try {
				res = AstonWizardsPlugin.getDefault().buildFromTemplate(tName);
			} catch (IOException e) {
				Utils.showError(e, getShell());
				return resu;
			}
			String pas = findXmlPart(res.toString(), MARK_PARENT_CLASS_BEGIN);
			if (pas != null) {
				resu = pas;
			}
			res = null;
		}
		return resu;
	}

	/**
	 * Generates the java file. 
	 * 
	 * @param type type of the object
	 * @param imports the imports of the file
	 * @param monitor the progress bar
	 * @param values a Map that contains every information needed for the creation of the class
	 * 
	 * @exception CoreException if an error occurred
	 */
	protected void createInternal(
		IType type,
		ImportsManager imports,
		IProgressMonitor monitor,
		Map values)
		throws CoreException {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeWizardPage.createInternal(type, imports, monitor, values)");
		StringBuffer fileSb = null;

		try {
			fileSb = AstonWizardsPlugin.getDefault().openContentStream(values);
		} catch (IOException e) {
			Utils.showError(e, getShell());
			return;
		}

		createInheritedMethods(
			type,
			false,
			isCreateInherited(),
			imports,
			monitor);

		String file = fileSb.toString();

		createImports(file, type, imports, monitor);
		createFields(file, type, imports, monitor);
		if (isCreateConstructors()) {
			createConstructor(file, type, imports, monitor);
		}
		createMethods(file, type, imports, monitor);
		createInerType(file, type, imports, monitor);

		updateSectionValues();

		if (monitor != null) {
			monitor.done();
		}
	}

	/**
	 * Memorize the selected options. <br>
	 */
	protected void updateSectionValues() {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.updateSectionValues()");
		IDialogSettings section =
			getDialogSettings().getSection(this.getName());
		if (section == null) {
			section = getDialogSettings().addNewSection(this.getName());
		}
		section.put(SETTINGS_CREATE_INHERITED, isCreateInherited());
		section.put(SETTINGS_CREATE_MAIN, isCreateMain());
		section.put(SETTINGS_CREATE_CONSTR, isCreateConstructors());
		section.put(SETTINGS_TEMPLATE, this.templateList.getText());		
	}

	/**
	 * Builds the fields for the given type.
	 * 
	 * @param file the String loaded from the template file
	 * @param type the created type
	 * @param imports object used for importing classes
	 * @param monitor the monitor used during the process
	 * 
	 * @exception CoreException if a problem occured
	 */
	protected void createFields(
		String file,
		IType type,
		ImportsManager imports,
		IProgressMonitor monitor)
		throws CoreException {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeWizardPage.createFields(file, type, imports, monitor)");
		Map m = loadXmlParts(file, MARK_FIELD, null);
		Iterator it = m.values().iterator();
		while (it.hasNext()) {
			String body = (String) it.next();
			type.createField(body, null, false, monitor);
			body = null;
		}
		it = null;
		m.clear();
		m = null;
	}

	/**
	 * Builds the methods for the given type.
	 * 
	 * @param file the String loaded from the template file
	 * @param type the created type
	 * @param imports object used for importing classes
	 * @param monitor the monitor used during the process
	 * 
	 * @exception CoreException if a problem occured
	 */
	protected void createMethods(
		String file,
		IType type,
		ImportsManager imports,
		IProgressMonitor monitor)
		throws CoreException {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeWizardPage.createMethods(file, type, imports, monitor)");
		Map m = loadXmlParts(file, MARK_METHOD, Marks.TOKEN_NAME);
		boolean doMain = isCreateMain();
		Iterator it = m.keySet().iterator();
		while (it.hasNext()) {
			String key = (String) it.next();
			String body = (String) m.get(key);
			if (!doMain && "main".equals(key)) {
				continue;
			}
			type.createMethod(body, null, false, monitor);
			body = null;
			key = null;
		}
		it = null;
		m.clear();
		m = null;
	}

	/**
	 * Builds all constructors for the given type.
	 * 
	 * @param file the String loaded from the template file
	 * @param type the created type
	 * @param imports object used for importing classes
	 * @param monitor the monitor used during the process
	 * 
	 * @exception CoreException if a problem occured
	 */
	protected void createConstructor(
		String file,
		IType type,
		ImportsManager imports,
		IProgressMonitor monitor)
		throws CoreException {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeWizardPage.createConstructor(file, type, imports, monitor)");
		Map m = loadXmlParts(file, MARK_CONSTRUCTOR, Marks.TOKEN_NAME);
		Iterator it = m.values().iterator();
		while (it.hasNext()) {
			String body = (String) it.next();
			type.createMethod(body, null, false, monitor);
			body = null;
		}
		it = null;
		m.clear();
		m = null;
	}

	/**
	 * Builds the iner type struct for the given type.
	 * 
	 * @param file the String loaded from the template file
	 * @param type the created type
	 * @param imports object used for importing classes
	 * @param monitor the monitor used during the process
	 * 
	 * @exception CoreException if a problem occured
	 */
	protected void createInerType(
		String file,
		IType type,
		ImportsManager imports,
		IProgressMonitor monitor)
		throws CoreException {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeWizardPage.createInerType(file, type, imports, monitor)");
		Map m = loadXmlParts(file, MARK_INERTYPE, Marks.TOKEN_NAME);
		Iterator it = m.values().iterator();
		while (it.hasNext()) {
			String body = (String) it.next();
			type.createType(body, null, false, monitor);
			body = null;
		}
		it = null;
		m.clear();
		m = null;
	}

	/**
	 * Builds the import for the given type.
	 * 
	 * @param file the String loaded from the template file
	 * @param type the created type
	 * @param imports object used for importing classes
	 * @param monitor the monitor used during the process
	 * 
	 * @exception CoreException if a problem occured
	 */
	protected void createImports(
		String file,
		IType type,
		ImportsManager imports,
		IProgressMonitor monitor)
		throws CoreException {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeWizardPage.createImports(file, type, imports, monitor)");
		Map m = loadXmlParts(file, MARK_IMPORT, null);
		Iterator it = m.values().iterator();
		while (it.hasNext()) {
			String body = (String) it.next();
			imports.addImport(body);
			body = null;
		}
		it = null;
		m.clear();
		m = null;
	}

	/**
	 * What to do if field change.
	 * 
	 * @param field who has changed
	 */
	public void dialogFieldChanged(DialogField field) {
		LogMessage.getInstance().log(
				"-> NewTypeWizardPage.dialogFieldChanged(field)");
		refreshSuperClass(field == this.templateList);
		refreshInterfaces(field == this.templateList);

		doStatusUpdate();
	}
}