/*
 * Copyright (c) 1997-2002 Genuitec, LLC.
 * All rights reserved.
 */
package com.aston.wizards.ejb;

import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.IBuffer;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.SelectionButtonDialogFieldGroup;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;

import com.aston.AstonWizardsPlugin;
import com.aston.LogMessage;

/**
 * Wizard page to create a new EJB class. <p>
 * 
 * This class is inspired by the org.eclipse.jdt.ui.wizards.NewClassWizardPage.
 * 
 * Part of the <a href="http://renaud91.free.fr/Plugins>Aston Wizard</a>
 * contributed by Genuitec, LLC <br>
 * 
 * @author Genuitec, LLC
 */
public class NewEjbCreationWizardPage extends com.aston.NewTypeWizardPage {
 
	private final static String SETTINGS_CREATE_REQUIRED_METHODS =
		"create_required";
	private final static String SETTINGS_CREATE_EXAMPLE_METHODS =
		"create_example";

	/**
	 * Constructor of the page. 
	 */
	public NewEjbCreationWizardPage() {
		super(true, "NewEjbCreationWizardPage", "ejb");
		LogMessage.getInstance().log(
				"-> NewEjbCreationWizardPage.NewEjbCreationWizardPage()");
	}

	/**
	 * The wizard owning this page is responsible for calling this method 
	 * with the current selection. The selection is used to 
	 * initialize the fields of the wizard page.
	 * 
	 * @param selection used to initialize the fields
	 */
	public void init(IStructuredSelection selection) {
		LogMessage.getInstance().log(
				"-> NewEjbCreationWizardPage.init(selection)");
		IJavaElement jelem = getInitialJavaElement(selection);
		initContainerPage(jelem);
		initTypePage(jelem);
		doStatusUpdate();

		boolean[] values = new boolean[] { true, true };

		IDialogSettings section =
			getDialogSettings().getSection(this.getName());
		if (section != null) {
			values[0] = section.getBoolean(SETTINGS_CREATE_REQUIRED_METHODS);
			values[1] = section.getBoolean(SETTINGS_CREATE_EXAMPLE_METHODS);
		}
		setMethodStubSelection(values, true);
	}

	/**
	 * Returns the current selection state of the 'Create required' checkbox.
	 * 
	 * @return the selection state of the 'Create required' checkbox
	 */
	public boolean isCreateRequired() {
		LogMessage.getInstance().log(
				"-> NewEjbCreationWizardPage.isCreateRequired()");
		return this.fMethodStubsButtons.isSelected(0);
	}

	/**
	 * Returns the current selection state of the 'Create example' checkbox.
	 * 
	 * @return the selection state of the 'Create example' checkbox
	 */
	public boolean isCreateExample() {
		LogMessage.getInstance().log(
				"-> NewEjbCreationWizardPage.isCreateExample()");
		return this.fMethodStubsButtons.isSelected(1);
	}

	protected void updateSectionValues() {
		LogMessage.getInstance().log(
				"-> NewEjbCreationWizardPage.updateSectionValues()");
		super.updateSectionValues();
		IDialogSettings section =
			getDialogSettings().getSection(this.getName());
		if (section == null) {
			section = getDialogSettings().addNewSection(this.getName());
		}

		section.put(SETTINGS_CREATE_REQUIRED_METHODS, isCreateRequired());
		section.put(SETTINGS_CREATE_EXAMPLE_METHODS, isCreateExample());
	}

	/**
	 * Builds the methods for the given type.
	 * 
	 * @param file the String loaded from the template file
	 * @param type the created type
	 * @param imports object used for importing classes
	 * @param monitor the monitor used during the process
	 * 
	 * @throws CoreException if a problem occured
	 */
	protected void createMethods(
		String file,
		IType type,
		ImportsManager imports,
		IProgressMonitor monitor)
		throws CoreException {
		LogMessage
				.getInstance()
				.log(
						"-> NewEjbCreationWizardPage.createMethods(file, type, imports, monitor)");
		Map m = loadXmlParts(file, MARK_METHOD, TOKEN_NAME);
		boolean required = isCreateRequired();
		boolean example = isCreateExample();

		/*
		 * delete methods generated by implementing specified interfaces
		 * so template versions with XDoclet comments can be inserted
		 */
		IMethod[] methods = type.getMethods();
		for (int i = 0; i < methods.length; i++) {
			methods[i].delete(true, monitor);
		}

		Iterator it = m.keySet().iterator();
		while (it.hasNext()) {
			String key = (String) it.next();
			String body = (String) m.get(key);
			if ((!required && key.indexOf("required") >= 0)
				|| (!example && key.indexOf("example") >= 0)) {
				continue;
			}
			type.createMethod(body, null, false, monitor);
			body = null;
			key = null;
		}

		it = null;
		m.clear();
		m = null;

		/*
		 * Get all the tags marked "class-comment", concatenate them
		 * together and add them in front of the class declaration.
		 */
		Map comments = loadXmlParts(file, "aw:class-comment", TOKEN_NAME);
		Iterator commentsIter = comments.keySet().iterator();
		StringBuffer commentsBuf = new StringBuffer(256);

		while (commentsIter.hasNext()) {
			commentsBuf.append((String) comments.get(commentsIter.next()));
		}

		IBuffer typeBuffer = type.getCompilationUnit().getBuffer();
		int publicIndex = typeBuffer.getContents().indexOf("\npublic ");
		typeBuffer.replace(publicIndex, 1, commentsBuf.toString());
	}

	/**
	 * @see com.aston.NewTypeWizardPage#initThePage()
	 */
	protected void initThePage() {
		LogMessage.getInstance().log(
				"-> NewEjbCreationWizardPage.initThePage()");
		setTitle(AstonWizardsPlugin.getResourceString("lg.label.wizejb"));
		setDescription(AstonWizardsPlugin.getResourceString("lg.label.ejbwillgen"));

		String[] buttonNames3 =
			new String[] {
				AstonWizardsPlugin.getResourceString("lg.label.createrequired"),
				AstonWizardsPlugin.getResourceString("lg.label.createexample"),
				};
		this.fMethodStubsButtons =
			new SelectionButtonDialogFieldGroup(SWT.CHECK, buttonNames3, 2);
		this.fMethodStubsButtons.setLabelText(
			AstonWizardsPlugin.getResourceString("lg.label.options"));
	}

	/**
	 * Adds specfic values to the Map of keys.
	 * 
	 * @param type type of the object
	 * @param imports the imports of the file
	 * @param monitor the progress bar
	 * @param values the current Map that contains values for the template
	 */
	protected void setSpecificValues(
		IType type,
		ImportsManager imports,
		IProgressMonitor monitor,
		Map values) {
		LogMessage
				.getInstance()
				.log(
						"-> NewEjbCreationWizardPage.setSpecificValues(type, imports, monitor, values)");
		// Does nothing
	}	
}